-- Base de datos para Boleto Feliz
CREATE DATABASE IF NOT EXISTS boleto_feliz CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE boleto_feliz;

-- Tabla de configuración del sistema
CREATE TABLE configuracion (
    id INT PRIMARY KEY AUTO_INCREMENT,
    total_boletos INT NOT NULL DEFAULT 5000,
    precio_boleto DECIMAL(10,2) NOT NULL DEFAULT 1.00,
    nombre_rifa VARCHAR(255) NOT NULL DEFAULT 'Rifa Boleto Feliz',
    fecha_sorteo DATE,
    estado_rifa ENUM('activa', 'pausada', 'finalizada') DEFAULT 'activa',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tabla de administradores
CREATE TABLE administradores (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabla de cuentas para depósito
CREATE TABLE cuentas_deposito (
    id INT PRIMARY KEY AUTO_INCREMENT,
    banco VARCHAR(100) NOT NULL,
    numero_cuenta VARCHAR(50) NOT NULL,
    titular VARCHAR(255) NOT NULL,
    tipo_cuenta ENUM('ahorros', 'corriente') DEFAULT 'ahorros',
    activa BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabla de clientes
CREATE TABLE clientes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nombre VARCHAR(255) NOT NULL,
    whatsapp VARCHAR(20) NOT NULL,
    email VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabla de pedidos
CREATE TABLE pedidos (
    id INT PRIMARY KEY AUTO_INCREMENT,
    cliente_id INT NOT NULL,
    total_boletos INT NOT NULL,
    monto_total DECIMAL(10,2) NOT NULL,
    estado ENUM('pendiente', 'pagado', 'cancelado') DEFAULT 'pendiente',
    fecha_limite DATE,
    comprobante_pago VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (cliente_id) REFERENCES clientes(id) ON DELETE CASCADE
);

-- Tabla de boletos
CREATE TABLE boletos (
    id INT PRIMARY KEY AUTO_INCREMENT,
    numero_boleto INT UNIQUE NOT NULL,
    pedido_id INT,
    cliente_id INT,
    estado ENUM('disponible', 'reservado', 'vendido') DEFAULT 'disponible',
    fecha_reserva TIMESTAMP NULL,
    fecha_venta TIMESTAMP NULL,
    es_ganador BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (pedido_id) REFERENCES pedidos(id) ON DELETE SET NULL,
    FOREIGN KEY (cliente_id) REFERENCES clientes(id) ON DELETE SET NULL,
    INDEX idx_numero_boleto (numero_boleto),
    INDEX idx_estado (estado)
);

-- Insertar configuración inicial
INSERT INTO configuracion (total_boletos, precio_boleto, nombre_rifa) VALUES (5000, 1.00, 'Rifa Boleto Feliz');

-- Insertar administrador por defecto (usuario: admin, password: admin123)
INSERT INTO administradores (username, password, email) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@boletofeliz.com');

-- Insertar cuentas de depósito de ejemplo
INSERT INTO cuentas_deposito (banco, numero_cuenta, titular, tipo_cuenta) VALUES 
('Banco Popular', '1234567890', 'Juan Pérez', 'ahorros'),
('Banco Central', '0987654321', 'María González', 'corriente');

-- Generar todos los boletos disponibles
INSERT INTO boletos (numero_boleto) 
SELECT n FROM (
    SELECT @row := @row + 1 as n FROM 
    (SELECT 0 UNION ALL SELECT 1 UNION ALL SELECT 2 UNION ALL SELECT 3 UNION ALL SELECT 4 UNION ALL SELECT 5 UNION ALL SELECT 6 UNION ALL SELECT 7 UNION ALL SELECT 8 UNION ALL SELECT 9) a,
    (SELECT 0 UNION ALL SELECT 1 UNION ALL SELECT 2 UNION ALL SELECT 3 UNION ALL SELECT 4 UNION ALL SELECT 5 UNION ALL SELECT 6 UNION ALL SELECT 7 UNION ALL SELECT 8 UNION ALL SELECT 9) b,
    (SELECT 0 UNION ALL SELECT 1 UNION ALL SELECT 2 UNION ALL SELECT 3 UNION ALL SELECT 4 UNION ALL SELECT 5 UNION ALL SELECT 6 UNION ALL SELECT 7 UNION ALL SELECT 8 UNION ALL SELECT 9) c,
    (SELECT 0 UNION ALL SELECT 1 UNION ALL SELECT 2 UNION ALL SELECT 3 UNION ALL SELECT 4 UNION ALL SELECT 5 UNION ALL SELECT 6 UNION ALL SELECT 7 UNION ALL SELECT 8 UNION ALL SELECT 9) d,
    (SELECT @row := 0) r
) numbers WHERE n BETWEEN 1 AND 5000;

-- Procedimientos almacenados útiles

-- Obtener estadísticas de la rifa
DELIMITER //
CREATE PROCEDURE ObtenerEstadisticas()
BEGIN
    SELECT 
        COUNT(*) as total_boletos,
        SUM(CASE WHEN estado = 'vendido' THEN 1 ELSE 0 END) as boletos_vendidos,
        SUM(CASE WHEN estado = 'reservado' THEN 1 ELSE 0 END) as boletos_reservados,
        SUM(CASE WHEN estado = 'disponible' THEN 1 ELSE 0 END) as boletos_disponibles,
        ROUND((SUM(CASE WHEN estado = 'vendido' THEN 1 ELSE 0 END) * 100.0 / COUNT(*)), 2) as porcentaje_vendidos
    FROM boletos;
END//

-- Reservar boletos para un pedido
CREATE PROCEDURE ReservarBoletos(IN p_cliente_id INT, IN p_pedido_id INT, IN p_numeros TEXT)
BEGIN
    DECLARE EXIT HANDLER FOR SQLEXCEPTION
    BEGIN
        ROLLBACK;
        RESIGNAL;
    END;

    START TRANSACTION;
    
    SET @sql = CONCAT('UPDATE boletos SET estado = "reservado", pedido_id = ', p_pedido_id, 
                     ', cliente_id = ', p_cliente_id, ', fecha_reserva = NOW() WHERE numero_boleto IN (', p_numeros, ') AND estado = "disponible"');
    
    PREPARE stmt FROM @sql;
    EXECUTE stmt;
    DEALLOCATE PREPARE stmt;
    
    COMMIT;
END//

-- Confirmar venta de boletos
CREATE PROCEDURE ConfirmarVenta(IN p_pedido_id INT)
BEGIN
    UPDATE boletos SET estado = 'vendido', fecha_venta = NOW() 
    WHERE pedido_id = p_pedido_id AND estado = 'reservado';
    
    UPDATE pedidos SET estado = 'pagado', updated_at = NOW() 
    WHERE id = p_pedido_id;
END//

DELIMITER ;