<?php
// api/pedido.php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Método no permitido'
    ]);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validar datos de entrada
    $nombre = trim($input['nombre'] ?? '');
    $whatsapp = trim($input['whatsapp'] ?? '');
    $email = trim($input['email'] ?? '');
    $boletos = $input['boletos'] ?? [];
    
    if (empty($nombre)) {
        echo json_encode([
            'success' => false,
            'message' => 'El nombre es requerido'
        ]);
        exit;
    }
    
    if (empty($whatsapp)) {
        echo json_encode([
            'success' => false,
            'message' => 'El número de WhatsApp es requerido'
        ]);
        exit;
    }
    
    if (empty($boletos) || !is_array($boletos)) {
        echo json_encode([
            'success' => false,
            'message' => 'Debe seleccionar al menos un boleto'
        ]);
        exit;
    }
    
    // Validar formato de WhatsApp
    $whatsapp = preg_replace('/[^0-9+]/', '', $whatsapp);
    if (strlen($whatsapp) < 10) {
        echo json_encode([
            'success' => false,
            'message' => 'Número de WhatsApp no válido'
        ]);
        exit;
    }
    
    // Validar email si se proporciona
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode([
            'success' => false,
            'message' => 'Correo electrónico no válido'
        ]);
        exit;
    }
    
    // Convertir boletos a enteros y validar
    $boletos = array_map('intval', $boletos);
    $boletos = array_filter($boletos, function($num) {
        return $num >= 1 && $num <= 5000;
    });
    
    if (empty($boletos)) {
        echo json_encode([
            'success' => false,
            'message' => 'Los números de boletos no son válidos'
        ]);
        exit;
    }
    
    // Eliminar duplicados
    $boletos = array_unique($boletos);
    
    $db = new DatabaseHelper();
    
    // Verificar disponibilidad de los boletos
    $disponibles = $db->verificarDisponibilidad($boletos);
    $noDisponibles = array_diff($boletos, $disponibles);
    
    if (!empty($noDisponibles)) {
        echo json_encode([
            'success' => false,
            'message' => 'Los siguientes boletos ya no están disponibles: ' . implode(', ', $noDisponibles)
        ]);
        exit;
    }
    
    // Obtener configuración para precio
    $config = $db->getConfig();
    $precioBoleto = floatval($config['precio_boleto'] ?? 1.00);
    
    $totalBoletos = count($boletos);
    $montoTotal = $totalBoletos * $precioBoleto;
    
    // Iniciar transacción
    $conn = $db->getConnection();
    $conn->beginTransaction();
    
    try {
        // Crear cliente
        $clienteId = $db->crearCliente($nombre, $whatsapp, $email);
        
        // Crear pedido
        $pedidoId = $db->crearPedido($clienteId, $totalBoletos, $montoTotal);
        
        // Reservar boletos
        $db->reservarBoletos($clienteId, $pedidoId, $boletos);
        
        // Confirmar transacción
        $conn->commit();
        
        // Respuesta exitosa
        echo json_encode([
            'success' => true,
            'message' => 'Pedido creado exitosamente',
            'data' => [
                'pedido_id' => $pedidoId,
                'cliente_id' => $clienteId,
                'nombre' => $nombre,
                'whatsapp' => $whatsapp,
                'email' => $email,
                'boletos' => implode(', ', $boletos),
                'total_boletos' => $totalBoletos,
                'monto_total' => number_format($montoTotal, 2),
                'fecha_limite' => date('Y-m-d', strtotime('+3 days'))
            ]
        ]);
        
    } catch (Exception $e) {
        $conn->rollback();
        throw $e;
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Error al procesar el pedido: ' . $e->getMessage()
    ]);
}
?>
                