<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Boleto Feliz - Sistema de Rifas</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #4f46e5;
            --primary-dark: #3730a3;
            --secondary-color: #06b6d4;
            --success-color: #059669;
            --warning-color: #d97706;
            --danger-color: #dc2626;
            --background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            --card-bg: rgba(255, 255, 255, 0.95);
            --text-dark: #1f2937;
            --text-light: #6b7280;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--background);
            min-height: 100vh;
            color: var(--text-dark);
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .header {
            text-align: center;
            margin-bottom: 3rem;
            padding: 2rem 0;
        }

        .header h1 {
            font-size: 3.5rem;
            font-weight: 700;
            color: white;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
            margin-bottom: 1rem;
        }

        .header p {
            font-size: 1.3rem;
            color: rgba(255,255,255,0.9);
            max-width: 600px;
            margin: 0 auto;
        }

        .progress-section {
            background: var(--card-bg);
            border-radius: 20px;
            padding: 2rem;
            margin-bottom: 3rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            backdrop-filter: blur(10px);
        }

        .progress-header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .progress-header h2 {
            color: var(--primary-color);
            margin-bottom: 0.5rem;
            font-size: 2rem;
        }

        .progress-bar-container {
            background: #e5e7eb;
            border-radius: 50px;
            height: 30px;
            overflow: hidden;
            margin-bottom: 1rem;
            position: relative;
        }

        .progress-bar {
            background: linear-gradient(90deg, var(--secondary-color), var(--primary-color));
            height: 100%;
            border-radius: 50px;
            transition: width 0.8s ease-in-out;
            position: relative;
            overflow: hidden;
        }

        .progress-bar::after {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
            animation: shimmer 2s infinite;
        }

        @keyframes shimmer {
            0% { left: -100%; }
            100% { left: 100%; }
        }

        .progress-text {
            text-align: center;
            font-weight: 600;
            color: var(--primary-color);
            font-size: 1.1rem;
        }

        .main-content {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 3rem;
            margin-bottom: 3rem;
        }

        .card {
            background: var(--card-bg);
            border-radius: 20px;
            padding: 2.5rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            backdrop-filter: blur(10px);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }

        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }

        .card h3 {
            color: var(--primary-color);
            margin-bottom: 1.5rem;
            font-size: 1.8rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--text-dark);
        }

        .form-control {
            width: 100%;
            padding: 0.875rem 1rem;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            font-size: 1rem;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
            background: white;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
        }

        .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.875rem 1.75rem;
            border: none;
            border-radius: 12px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            transition: all 0.3s ease;
            justify-content: center;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, var(--primary-dark), var(--primary-color));
            transform: translateY(-2px);
            box-shadow: 0 10px 20px -5px rgba(79, 70, 229, 0.4);
        }

        .btn-secondary {
            background: linear-gradient(135deg, var(--secondary-color), #0891b2);
            color: white;
        }

        .btn-secondary:hover {
            background: linear-gradient(135deg, #0891b2, var(--secondary-color));
            transform: translateY(-2px);
            box-shadow: 0 10px 20px -5px rgba(6, 182, 212, 0.4);
        }

        .btn-block {
            width: 100%;
        }

        .selection-methods {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            margin-bottom: 1.5rem;
        }

        .method-card {
            padding: 1rem;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .method-card.active {
            border-color: var(--primary-color);
            background: rgba(79, 70, 229, 0.05);
        }

        .method-card:hover {
            border-color: var(--primary-color);
        }

        .method-card i {
            font-size: 2rem;
            margin-bottom: 0.5rem;
            color: var(--primary-color);
        }

        .boletos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(50px, 1fr));
            gap: 0.5rem;
            max-height: 300px;
            overflow-y: auto;
            padding: 1rem;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            background: #f9fafb;
        }

        .boleto-item {
            aspect-ratio: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            border: 2px solid #d1d5db;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            font-size: 0.8rem;
            transition: all 0.2s ease;
            background: white;
        }

        .boleto-item.disponible:hover {
            border-color: var(--primary-color);
            background: rgba(79, 70, 229, 0.1);
        }

        .boleto-item.seleccionado {
            background: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
        }

        .boleto-item.vendido {
            background: #fca5a5;
            color: #7f1d1d;
            cursor: not-allowed;
            border-color: #fca5a5;
        }

        .seleccionados-info {
            background: rgba(79, 70, 229, 0.1);
            border-radius: 12px;
            padding: 1rem;
            margin-top: 1rem;
        }

        .admin-link {
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: rgba(0,0,0,0.7);
            color: white;
            padding: 1rem;
            border-radius: 50%;
            text-decoration: none;
            font-size: 1.2rem;
            transition: all 0.3s ease;
        }

        .admin-link:hover {
            background: rgba(0,0,0,0.9);
            transform: scale(1.1);
        }

        @media (max-width: 768px) {
            .container {
                padding: 15px;
            }

            .header h1 {
                font-size: 2.5rem;
            }

            .main-content {
                grid-template-columns: 1fr;
                gap: 2rem;
            }

            .selection-methods {
                grid-template-columns: 1fr;
            }

            .card {
                padding: 1.5rem;
            }

            .boletos-grid {
                grid-template-columns: repeat(auto-fill, minmax(40px, 1fr));
                max-height: 250px;
            }
        }

        .loading {
            display: none;
            text-align: center;
            padding: 2rem;
        }

        .spinner {
            width: 40px;
            height: 40px;
            border: 4px solid #e5e7eb;
            border-top: 4px solid var(--primary-color);
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 1rem;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .alert {
            padding: 1rem;
            border-radius: 12px;
            margin-bottom: 1rem;
        }

        .alert-success {
            background: rgba(5, 150, 105, 0.1);
            color: var(--success-color);
            border: 1px solid rgba(5, 150, 105, 0.2);
        }

        .alert-error {
            background: rgba(220, 38, 38, 0.1);
            color: var(--danger-color);
            border: 1px solid rgba(220, 38, 38, 0.2);
        }

        .precio-info {
            background: rgba(6, 182, 212, 0.1);
            border-radius: 12px;
            padding: 1rem;
            text-align: center;
            margin-bottom: 1rem;
        }

        .precio-info .precio {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--secondary-color);
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <div class="header">
            <h1><i class="fas fa-ticket-alt"></i> Boleto Feliz</h1>
            <p>Participa en nuestra rifa y gana increíbles premios. ¡Elige tus números de la suerte!</p>
        </div>

        <!-- Progress Section -->
        <div class="progress-section">
            <div class="progress-header">
                <h2><i class="fas fa-chart-line"></i> Progreso de Ventas</h2>
                <p>Boletos vendidos hasta el momento</p>
            </div>
            <div class="progress-bar-container">
                <div class="progress-bar" id="progressBar" style="width: 0%"></div>
            </div>
            <div class="progress-text" id="progressText">0% de boletos vendidos</div>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <!-- Selección de Boletos -->
            <div class="card">
                <h3><i class="fas fa-hand-pointer"></i> Seleccionar Boletos</h3>
                
                <div class="precio-info">
                    <div>Precio por boleto: <span class="precio">$1.00</span></div>
                </div>

                <div class="selection-methods">
                    <div class="method-card active" onclick="cambiarModo('manual')">
                        <i class="fas fa-hand-pointer"></i>
                        <div>Selección Manual</div>
                    </div>
                    <div class="method-card" onclick="cambiarModo('aleatorio')">
                        <i class="fas-random"></i>
                        <div>Selección Aleatoria</div>
                    </div>
                </div>

                <!-- Selección Manual -->
                <div id="seleccionManual">
                    <div class="boletos-grid" id="boletosGrid">
                        <!-- Los boletos se cargarán aquí dinámicamente -->
                    </div>
                </div>

                <!-- Selección Aleatoria -->
                <div id="seleccionAleatoria" style="display: none;">
                    <div class="form-group">
                        <label for="cantidadBoletos">Cantidad de boletos a generar:</label>
                        <input type="number" id="cantidadBoletos" class="form-control" min="1" max="100" placeholder="Ej: 5">
                    </div>
                    <button class="btn btn-secondary btn-block" onclick="generarAleatorios()">
                        <i class="fas fa-dice"></i> Generar Números Aleatorios
                    </button>
                </div>

                <div class="seleccionados-info" id="seleccionadosInfo" style="display: none;">
                    <h4><i class="fas fa-list"></i> Boletos Seleccionados</h4>
                    <div id="listaSeleccionados"></div>
                    <div style="margin-top: 1rem;">
                        <strong>Total: <span id="totalBoletos">0</span> boletos = $<span id="totalPrecio">0.00</span></strong>
                    </div>
                    <button class="btn btn-primary btn-block" onclick="continuarPedido()" style="margin-top: 1rem;">
                        <i class="fas fa-arrow-right"></i> Continuar con el Pedido
                    </button>
                </div>
            </div>

            <!-- Información del Cliente -->
            <div class="card" id="informacionCliente" style="display: none;">
                <h3><i class="fas fa-user"></i> Información del Cliente</h3>
                
                <div class="form-group">
                    <label for="nombreCliente">Nombre Completo *</label>
                    <input type="text" id="nombreCliente" class="form-control" required placeholder="Tu nombre completo">
                </div>

                <div class="form-group">
                    <label for="whatsappCliente">Número de WhatsApp *</label>
                    <input type="tel" id="whatsappCliente" class="form-control" required placeholder="+593 999 999 999">
                </div>

                <div class="form-group">
                    <label for="emailCliente">Correo Electrónico</label>
                    <input type="email" id="emailCliente" class="form-control" placeholder="tu@email.com">
                </div>

                <div id="resumenPedido" class="seleccionados-info">
                    <h4><i class="fas fa-receipt"></i> Resumen del Pedido</h4>
                    <div id="resumenBoletos"></div>
                    <div id="resumenTotal"></div>
                </div>

                <button class="btn btn-primary btn-block" onclick="completarPedido()">
                    <i class="fas fa-shopping-cart"></i> Completar Pedido
                </button>

                <button class="btn btn-secondary btn-block" onclick="volverSeleccion()" style="margin-top: 0.5rem;">
                    <i class="fas fa-arrow-left"></i> Volver a Selección
                </button>
            </div>

            <!-- Información de Depósito -->
            <div class="card" id="informacionDeposito" style="display: none;">
                <h3><i class="fas fa-credit-card"></i> Información de Pago</h3>
                
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    ¡Pedido creado exitosamente! Realiza el depósito en cualquiera de las siguientes cuentas:
                </div>

                <div id="cuentasDeposito">
                    <!-- Las cuentas se cargarán aquí dinámicamente -->
                </div>

                <div class="seleccionados-info">
                    <h4><i class="fas fa-info-circle"></i> Detalles de tu Pedido</h4>
                    <div id="detallesPedido"></div>
                    <div style="margin-top: 1rem;">
                        <strong>Total a Depositar: $<span id="montoDeposito">0.00</span></strong>
                    </div>
                </div>

                <div class="alert" style="background: rgba(217, 119, 6, 0.1); color: var(--warning-color); border: 1px solid rgba(217, 119, 6, 0.2);">
                    <i class="fas fa-clock"></i>
                    <strong>Importante:</strong> Tienes 3 días para realizar el depósito. Después de este tiempo, tu reserva se cancelará automáticamente.
                </div>

                <button class="btn btn-primary btn-block" onclick="nuevoPedido()">
                    <i class="fas fa-plus"></i> Realizar Nuevo Pedido
                </button>
            </div>
        </div>

        <!-- Loading -->
        <div class="loading" id="loading">
            <div class="spinner"></div>
            <p>Procesando...</p>
        </div>
    </div>

    <!-- Admin Link -->
    <a href="admin-login.php" class="admin-link" title="Acceso Administrador">
        <i class="fas fa-cog"></i>
    </a>

    <script>
        // Variables globales
        let boletosDisponibles = [];
        let boletosSeleccionados = [];
        let modoSeleccion = 'manual';
        let estadisticas = {};

        // Inicializar la página
        document.addEventListener('DOMContentLoaded', function() {
            cargarEstadisticas();
            cargarBoletos();
        });

        // Cargar estadísticas
        async function cargarEstadisticas() {
            try {
                const response = await fetch('api/estadisticas.php');
                const data = await response.json();
                
                if (data.success) {
                    estadisticas = data.data;
                    actualizarProgreso();
                }
            } catch (error) {
                console.error('Error al cargar estadísticas:', error);
            }
        }

        // Actualizar barra de progreso
        function actualizarProgreso() {
            const porcentaje = estadisticas.porcentaje_vendidos || 0;
            const progressBar = document.getElementById('progressBar');
            const progressText = document.getElementById('progressText');
            
            progressBar.style.width = porcentaje + '%';
            progressText.textContent = porcentaje + '% de boletos vendidos';
        }

        // Cargar boletos disponibles
        async function cargarBoletos() {
            mostrarCargando(true);
            try {
                const response = await fetch('api/boletos.php?action=disponibles&limit=500');
                const data = await response.json();
                
                if (data.success) {
                    boletosDisponibles = data.data;
                    renderizarBoletos();
                }
            } catch (error) {
                console.error('Error al cargar boletos:', error);
                mostrarAlerta('Error al cargar boletos. Por favor, recarga la página.', 'error');
            }
            mostrarCargando(false);
        }

        // Renderizar boletos en la grid
        function renderizarBoletos() {
            const grid = document.getElementById('boletosGrid');
            grid.innerHTML = '';
            
            // Mostrar los primeros 500 boletos disponibles
            const boletosAMostrar = boletosDisponibles.slice(0, 500);
            
            boletosAMostrar.forEach(numero => {
                const boleto = document.createElement('div');
                boleto.className = 'boleto-item disponible';
                boleto.textContent = numero;
                boleto.onclick = () => seleccionarBoleto(numero);
                grid.appendChild(boleto);
            });

            // Si hay más boletos disponibles, mostrar mensaje
            if (boletosDisponibles.length > 500) {
                const mensaje = document.createElement('div');
                mensaje.style.gridColumn = '1 / -1';
                mensaje.style.textAlign = 'center';
                mensaje.style.padding = '1rem';
                mensaje.style.background = 'rgba(6, 182, 212, 0.1)';
                mensaje.style.borderRadius = '8px';
                mensaje.style.marginTop = '1rem';
                mensaje.innerHTML = `<i class="fas fa-info-circle"></i> Hay ${boletosDisponibles.length - 500} boletos más disponibles. Usa la búsqueda o selección aleatoria para acceder a todos.`;
                grid.appendChild(mensaje);
            }
        }

        // Cambiar modo de selección
        function cambiarModo(modo) {
            modoSeleccion = modo;
            
            // Actualizar cards activos
            document.querySelectorAll('.method-card').forEach(card => {
                card.classList.remove('active');
            });
            event.target.closest('.method-card').classList.add('active');
            
            // Mostrar/ocultar secciones
            if (modo === 'manual') {
                document.getElementById('seleccionManual').style.display = 'block';
                document.getElementById('seleccionAleatoria').style.display = 'none';
            } else {
                document.getElementById('seleccionManual').style.display = 'none';
                document.getElementById('seleccionAleatoria').style.display = 'block';
            }
        }

        // Seleccionar boleto manualmente
        function seleccionarBoleto(numero) {
            const index = boletosSeleccionados.indexOf(numero);
            
            if (index === -1) {
                // Agregar boleto
                boletosSeleccionados.push(numero);
            } else {
                // Quitar boleto
                boletosSeleccionados.splice(index, 1);
            }
            
            actualizarVisualizacionSeleccionados();
        }

        // Generar números aleatorios
        async function generarAleatorios() {
            const cantidad = parseInt(document.getElementById('cantidadBoletos').value);
            
            if (!cantidad || cantidad < 1 || cantidad > 100) {
                mostrarAlerta('Por favor, ingresa una cantidad válida (1-100)', 'error');
                return;
            }

            mostrarCargando(true);
            
            try {
                const response = await fetch(`api/boletos.php?action=aleatorios&cantidad=${cantidad}`);
                const data = await response.json();
                
                if (data.success) {
                    boletosSeleccionados = data.data;
                    actualizarVisualizacionSeleccionados();
                    mostrarAlerta(`Se generaron ${cantidad} números aleatorios exitosamente`, 'success');
                } else {
                    mostrarAlerta(data.message, 'error');
                }
            } catch (error) {
                console.error('Error al generar números aleatorios:', error);
                mostrarAlerta('Error al generar números aleatorios', 'error');
            }
            
            mostrarCargando(false);
        }

        // Actualizar visualización de boletos seleccionados
        function actualizarVisualizacionSeleccionados() {
            const info = document.getElementById('seleccionadosInfo');
            const lista = document.getElementById('listaSeleccionados');
            const totalBoletos = document.getElementById('totalBoletos');
            const totalPrecio = document.getElementById('totalPrecio');
            
            if (boletosSeleccionados.length === 0) {
                info.style.display = 'none';
                return;
            }
            
            info.style.display = 'block';
            
            // Ordenar y mostrar números
            boletosSeleccionados.sort((a, b) => a - b);
            lista.innerHTML = boletosSeleccionados.join(', ');
            
            // Actualizar totales
            totalBoletos.textContent = boletosSeleccionados.length;
            totalPrecio.textContent = (boletosSeleccionados.length * 1.00).toFixed(2);
            
            // Actualizar visualización en grid si está en modo manual
            if (modoSeleccion === 'manual') {
                document.querySelectorAll('.boleto-item').forEach(item => {
                    const numero = parseInt(item.textContent);
                    if (boletosSeleccionados.includes(numero)) {
                        item.classList.add('seleccionado');
                    } else {
                        item.classList.remove('seleccionado');
                    }
                });
            }
        }

        // Continuar con el pedido
        function continuarPedido() {
            if (boletosSeleccionados.length === 0) {
                mostrarAlerta('Por favor, selecciona al menos un boleto', 'error');
                return;
            }
            
            // Mostrar formulario de cliente
            document.getElementById('informacionCliente').style.display = 'block';
            
            // Actualizar resumen
            const resumenBoletos = document.getElementById('resumenBoletos');
            const resumenTotal = document.getElementById('resumenTotal');
            
            resumenBoletos.innerHTML = `
                <p><strong>Boletos seleccionados:</strong> ${boletosSeleccionados.join(', ')}</p>
            `;
            
            resumenTotal.innerHTML = `
                <p><strong>Total: ${boletosSeleccionados.length} boletos × $1.00 = ${(boletosSeleccionados.length * 1.00).toFixed(2)}</strong></p>
            `;
            
            // Scroll to form
            document.getElementById('informacionCliente').scrollIntoView({ behavior: 'smooth' });
        }

        // Volver a selección
        function volverSeleccion() {
            document.getElementById('informacionCliente').style.display = 'none';
            document.getElementById('informacionDeposito').style.display = 'none';
        }

        // Completar pedido
        async function completarPedido() {
            const nombre = document.getElementById('nombreCliente').value.trim();
            const whatsapp = document.getElementById('whatsappCliente').value.trim();
            const email = document.getElementById('emailCliente').value.trim();
            
            if (!nombre || !whatsapp) {
                mostrarAlerta('Por favor, completa los campos obligatorios', 'error');
                return;
            }
            
            mostrarCargando(true);
            
            try {
                const response = await fetch('api/pedido.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        nombre: nombre,
                        whatsapp: whatsapp,
                        email: email,
                        boletos: boletosSeleccionados
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    mostrarInformacionDeposito(data.data);
                } else {
                    mostrarAlerta(data.message, 'error');
                }
            } catch (error) {
                console.error('Error al crear pedido:', error);
                mostrarAlerta('Error al procesar el pedido', 'error');
            }
            
            mostrarCargando(false);
        }

        // Mostrar información de depósito
        async function mostrarInformacionDeposito(pedidoData) {
            // Ocultar formulario de cliente
            document.getElementById('informacionCliente').style.display = 'none';
            
            // Mostrar información de depósito
            document.getElementById('informacionDeposito').style.display = 'block';
            
            // Cargar cuentas de depósito
            try {
                const response = await fetch('api/cuentas.php');
                const data = await response.json();
                
                if (data.success) {
                    const cuentasContainer = document.getElementById('cuentasDeposito');
                    cuentasContainer.innerHTML = '';
                    
                    data.data.forEach(cuenta => {
                        const cuentaDiv = document.createElement('div');
                        cuentaDiv.className = 'seleccionados-info';
                        cuentaDiv.style.marginBottom = '1rem';
                        cuentaDiv.innerHTML = `
                            <h4><i class="fas fa-university"></i> ${cuenta.banco}</h4>
                            <p><strong>Número de Cuenta:</strong> ${cuenta.numero_cuenta}</p>
                            <p><strong>Titular:</strong> ${cuenta.titular}</p>
                            <p><strong>Tipo:</strong> ${cuenta.tipo_cuenta}</p>
                        `;
                        cuentasContainer.appendChild(cuentaDiv);
                    });
                }
            } catch (error) {
                console.error('Error al cargar cuentas:', error);
            }
            
            // Actualizar detalles del pedido
            const detallesPedido = document.getElementById('detallesPedido');
            const montoDeposito = document.getElementById('montoDeposito');
            
            detallesPedido.innerHTML = `
                <p><strong>Pedido #:</strong> ${pedidoData.pedido_id}</p>
                <p><strong>Cliente:</strong> ${pedidoData.nombre}</p>
                <p><strong>WhatsApp:</strong> ${pedidoData.whatsapp}</p>
                <p><strong>Boletos:</strong> ${pedidoData.boletos}</p>
                <p><strong>Cantidad:</strong> ${pedidoData.total_boletos} boletos</p>
            `;
            
            montoDeposito.textContent = pedidoData.monto_total;
            
            // Scroll to deposit info
            document.getElementById('informacionDeposito').scrollIntoView({ behavior: 'smooth' });
            
            // Actualizar estadísticas
            cargarEstadisticas();
        }

        // Nuevo pedido
        function nuevoPedido() {
            // Reset variables
            boletosSeleccionados = [];
            
            // Reset forms
            document.getElementById('nombreCliente').value = '';
            document.getElementById('whatsappCliente').value = '';
            document.getElementById('emailCliente').value = '';
            document.getElementById('cantidadBoletos').value = '';
            
            // Hide sections
            document.getElementById('informacionCliente').style.display = 'none';
            document.getElementById('informacionDeposito').style.display = 'none';
            document.getElementById('seleccionadosInfo').style.display = 'none';
            
            // Reset selection mode
            cambiarModo('manual');
            document.querySelectorAll('.method-card')[0].classList.add('active');
            
            // Reload data
            cargarEstadisticas();
            cargarBoletos();
            
            // Scroll to top
            window.scrollTo({ top: 0, behavior: 'smooth' });
        }

        // Funciones de utilidad
        function mostrarCargando(mostrar) {
            document.getElementById('loading').style.display = mostrar ? 'block' : 'none';
        }

        function mostrarAlerta(mensaje, tipo) {
            // Crear elemento de alerta
            const alerta = document.createElement('div');
            alerta.className = `alert alert-${tipo === 'success' ? 'success' : 'error'}`;
            alerta.innerHTML = `
                <i class="fas fa-${tipo === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
                ${mensaje}
            `;
            
            // Insertar al principio del container
            const container = document.querySelector('.container');
            container.insertBefore(alerta, container.firstChild);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                if (alerta.parentNode) {
                    alerta.parentNode.removeChild(alerta);
                }
            }, 5000);
            
            // Scroll to alert
            alerta.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    </script>
</body>
</html>