<?php
// admin-dashboard.php
require_once 'config/database.php';
requiereAdmin();

$db = new DatabaseHelper();
$config = $db->getConfig();
$estadisticas = $db->getEstadisticas();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panel de Administración - Boleto Feliz</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #4f46e5;
            --primary-dark: #3730a3;
            --secondary-color: #06b6d4;
            --success-color: #059669;
            --warning-color: #d97706;
            --danger-color: #dc2626;
            --background: #f3f4f6;
            --card-bg: white;
            --text-dark: #1f2937;
            --text-light: #6b7280;
            --border-color: #e5e7eb;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--background);
            color: var(--text-dark);
            line-height: 1.6;
        }

        .admin-header {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            padding: 1.5rem 0;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-title {
            font-size: 1.8rem;
            font-weight: 700;
        }

        .header-actions {
            display: flex;
            gap: 1rem;
            align-items: center;
        }

        .container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 0 20px;
        }

        .dashboard-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 2rem;
            margin-bottom: 3rem;
        }

        .stat-card {
            background: var(--card-bg);
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px -3px rgba(0, 0, 0, 0.1);
        }

        .stat-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 1rem;
        }

        .stat-title {
            font-size: 0.9rem;
            font-weight: 600;
            color: var(--text-light);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .stat-icon {
            width: 40px;
            height: 40px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
            color: white;
        }

        .stat-icon.blue { background: var(--primary-color); }
        .stat-icon.green { background: var(--success-color); }
        .stat-icon.orange { background: var(--warning-color); }
        .stat-icon.red { background: var(--danger-color); }

        .stat-value {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--text-dark);
            margin-bottom: 0.5rem;
        }

        .stat-label {
            font-size: 0.9rem;
            color: var(--text-light);
        }

        .admin-sections {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 3rem;
            margin-bottom: 3rem;
        }

        .admin-card {
            background: var(--card-bg);
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .admin-card h3 {
            color: var(--primary-color);
            margin-bottom: 1.5rem;
            font-size: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--text-dark);
        }

        .form-control {
            width: 100%;
            padding: 0.875rem 1rem;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            font-size: 1rem;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
            background: white;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
        }

        .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.875rem 1.75rem;
            border: none;
            border-radius: 12px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            transition: all 0.3s ease;
            justify-content: center;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, var(--primary-dark), var(--primary-color));
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(79, 70, 229, 0.4);
        }

        .btn-danger {
            background: var(--danger-color);
            color: white;
        }

        .btn-danger:hover {
            background: #b91c1c;
            transform: translateY(-1px);
        }

        .btn-secondary {
            background: var(--text-light);
            color: white;
        }

        .btn-secondary:hover {
            background: #4b5563;
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
        }

        .progress-bar-admin {
            background: var(--border-color);
            border-radius: 8px;
            height: 20px;
            overflow: hidden;
            margin-bottom: 1rem;
        }

        .progress-fill {
            background: linear-gradient(90deg, var(--secondary-color), var(--primary-color));
            height: 100%;
            transition: width 0.8s ease;
            border-radius: 8px;
        }

        .table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
        }

        .table th,
        .table td {
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }

        .table th {
            background: #f9fafb;
            font-weight: 600;
            color: var(--text-dark);
        }

        .badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }

        .badge-success {
            background: rgba(5, 150, 105, 0.1);
            color: var(--success-color);
        }

        .badge-warning {
            background: rgba(217, 119, 6, 0.1);
            color: var(--warning-color);
        }

        .badge-danger {
            background: rgba(220, 38, 38, 0.1);
            color: var(--danger-color);
        }

        .alert {
            padding: 1rem;
            border-radius: 12px;
            margin-bottom: 1rem;
        }

        .alert-success {
            background: rgba(5, 150, 105, 0.1);
            color: var(--success-color);
            border: 1px solid rgba(5, 150, 105, 0.2);
        }

        .alert-error {
            background: rgba(220, 38, 38, 0.1);
            color: var(--danger-color);
            border: 1px solid rgba(220, 38, 38, 0.2);
        }

        @media (max-width: 768px) {
            .admin-sections {
                grid-template-columns: 1fr;
                gap: 2rem;
            }

            .dashboard-grid {
                grid-template-columns: 1fr;
            }

            .header-content {
                flex-direction: column;
                gap: 1rem;
                text-align: center;
            }

            .header-actions {
                justify-content: center;
            }
        }

        .loading {
            display: none;
            text-align: center;
            padding: 2rem;
        }

        .spinner {
            width: 40px;
            height: 40px;
            border: 4px solid var(--border-color);
            border-top: 4px solid var(--primary-color);
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 1rem;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .full-width {
            grid-column: 1 / -1;
        }
    </style>
</head>
<body>
    <div class="admin-header">
        <div class="header-content">
            <div class="header-title">
                <i class="fas fa-tachometer-alt"></i> Panel de Administración
            </div>
            <div class="header-actions">
                <span>Bienvenido, <?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
                <a href="index.php" class="btn btn-secondary btn-sm">
                    <i class="fas fa-home"></i> Ver Sitio
                </a>
                <a href="api/admin-logout.php" class="btn btn-danger btn-sm">
                    <i class="fas fa-sign-out-alt"></i> Salir
                </a>
            </div>
        </div>
    </div>

    <div class="container">
        <!-- Estadísticas -->
        <div class="dashboard-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-title">Total Boletos</div>
                    <div class="stat-icon blue">
                        <i class="fas fa-ticket-alt"></i>
                    </div>
                </div>
                <div class="stat-value"><?php echo number_format($estadisticas['total_boletos']); ?></div>
                <div class="stat-label">Boletos en el sistema</div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-title">Vendidos</div>
                    <div class="stat-icon green">
                        <i class="fas fa-check-circle"></i>
                    </div>
                </div>
                <div class="stat-value"><?php echo number_format($estadisticas['boletos_vendidos']); ?></div>
                <div class="stat-label"><?php echo number_format($estadisticas['porcentaje_vendidos'], 2); ?>% del total</div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-title">Reservados</div>
                    <div class="stat-icon orange">
                        <i class="fas fa-clock"></i>
                    </div>
                </div>
                <div class="stat-value"><?php echo number_format($estadisticas['boletos_reservados']); ?></div>
                <div class="stat-label">Esperando pago</div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-title">Disponibles</div>
                    <div class="stat-icon red">
                        <i class="fas fa-circle"></i>
                    </div>
                </div>
                <div class="stat-value"><?php echo number_format($estadisticas['boletos_disponibles']); ?></div>
                <div class="stat-label">Listos para venta</div>
            </div>
        </div>

        <!-- Barra de progreso detallada -->
        <div class="admin-card full-width" style="margin-bottom: 3rem;">
            <h3><i class="fas fa-chart-line"></i> Progreso Detallado de Ventas</h3>
            
            <div class="progress-bar-admin">
                <div class="progress-fill" style="width: <?php echo $estadisticas['porcentaje_vendidos']; ?>%"></div>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; text-align: center;">
                <div>
                    <div style="font-size: 1.5rem; font-weight: 700; color: var(--success-color);">
                        <?php echo number_format($estadisticas['boletos_vendidos']); ?>
                    </div>
                    <div style="color: var(--text-light);">Vendidos</div>
                </div>
                <div>
                    <div style="font-size: 1.5rem; font-weight: 700; color: var(--warning-color);">
                        <?php echo number_format($estadisticas['boletos_reservados']); ?>
                    </div>
                    <div style="color: var(--text-light);">Reservados</div>
                </div>
                <div>
                    <div style="font-size: 1.5rem; font-weight: 700; color: var(--danger-color);">
                        <?php echo number_format($estadisticas['boletos_disponibles']); ?>
                    </div>
                    <div style="color: var(--text-light);">Disponibles</div>
                </div>
                <div>
                    <div style="font-size: 1.5rem; font-weight: 700; color: var(--primary-color);">
                        <?php echo number_format($estadisticas['porcentaje_vendidos'], 1); ?>%
                    </div>
                    <div style="color: var(--text-light);">Progreso</div>
                </div>
            </div>
        </div>

        <!-- Secciones de administración -->
        <div class="admin-sections">
            <!-- Configuración del sistema -->
            <div class="admin-card">
                <h3><i class="fas fa-cog"></i> Configuración del Sistema</h3>
                
                <form id="configForm">
                    <div class="form-group">
                        <label for="totalBoletos">Total de Boletos</label>
                        <input type="number" id="totalBoletos" class="form-control" 
                               value="<?php echo $config['total_boletos']; ?>" min="1000" max="10000">
                    </div>

                    <div class="form-group">
                        <label for="precioBoleto">Precio por Boleto (USD)</label>
                        <input type="number" id="precioBoleto" class="form-control" 
                               value="<?php echo $config['precio_boleto']; ?>" min="0.50" max="100" step="0.01">
                    </div>

                    <div class="form-group">
                        <label for="nombreRifa">Nombre de la Rifa</label>
                        <input type="text" id="nombreRifa" class="form-control" 
                               value="<?php echo htmlspecialchars($config['nombre_rifa']); ?>">
                    </div>

                    <div class="form-group">
                        <label for="fechaSorteo">Fecha del Sorteo</label>
                        <input type="date" id="fechaSorteo" class="form-control" 
                               value="<?php echo $config['fecha_sorteo']; ?>">
                    </div>

                    <div class="form-group">
                        <label for="estadoRifa">Estado de la Rifa</label>
                        <select id="estadoRifa" class="form-control">
                            <option value="activa" <?php echo $config['estado_rifa'] === 'activa' ? 'selected' : ''; ?>>Activa</option>
                            <option value="pausada" <?php echo $config['estado_rifa'] === 'pausada' ? 'selected' : ''; ?>>Pausada</option>
                            <option value="finalizada" <?php echo $config['estado_rifa'] === 'finalizada' ? 'selected' : ''; ?>>Finalizada</option>
                        </select>
                    </div>

                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Guardar Configuración
                    </button>
                </form>
            </div>

            <!-- Gestión de cuentas bancarias -->
            <div class="admin-card">
                <h3><i class="fas fa-university"></i> Cuentas Bancarias</h3>
                
                <form id="cuentaForm">
                    <div class="form-group">
                        <label for="banco">Banco</label>
                        <input type="text" id="banco" class="form-control" placeholder="Nombre del banco">
                    </div>

                    <div class="form-group">
                        <label for="numeroCuenta">Número de Cuenta</label>
                        <input type="text" id="numeroCuenta" class="form-control" placeholder="Número de cuenta">
                    </div>

                    <div class="form-group">
                        <label for="titular">Titular</label>
                        <input type="text" id="titular" class="form-control" placeholder="Nombre del titular">
                    </div>

                    <div class="form-group">
                        <label for="tipoCuenta">Tipo de Cuenta</label>
                        <select id="tipoCuenta" class="form-control">
                            <option value="ahorros">Ahorros</option>
                            <option value="corriente">Corriente</option>
                        </select>
                    </div>

                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Agregar Cuenta
                    </button>
                </form>

                <div id="cuentasList" class="table-container" style="margin-top: 2rem;">
                    <!-- Las cuentas se cargarán aquí -->
                </div>
            </div>
        </div>

        <!-- Pedidos recientes -->
        <div class="admin-card full-width">
            <h3><i class="fas fa-shopping-cart"></i> Pedidos Recientes</h3>
            
            <div class="table-container">
                <table class="table" id="pedidosTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Cliente</th>
                            <th>WhatsApp</th>
                            <th>Boletos</th>
                            <th>Monto</th>
                            <th>Estado</th>
                            <th>Fecha</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Los pedidos se cargarán aquí -->
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Loading -->
        <div class="loading" id="loading">
            <div class="spinner"></div>
            <p>Procesando...</p>
        </div>
    </div>

    <script>
        // Inicializar dashboard
        document.addEventListener('DOMContentLoaded', function() {
            cargarCuentas();
            cargarPedidos();
        });

        // Configuración del sistema
        document.getElementById('configForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = {
                total_boletos: parseInt(document.getElementById('totalBoletos').value),
                precio_boleto: parseFloat(document.getElementById('precioBoleto').value),
                nombre_rifa: document.getElementById('nombreRifa').value.trim(),
                fecha_sorteo: document.getElementById('fechaSorteo').value,
                estado_rifa: document.getElementById('estadoRifa').value
            };
            
            if (!formData.nombre_rifa) {
                mostrarAlerta('El nombre de la rifa es requerido', 'error');
                return;
            }
            
            mostrarCargando(true);
            
            try {
                const response = await fetch('api/admin-config.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(formData)
                });
                
                const data = await response.json();
                
                if (data.success) {
                    mostrarAlerta('Configuración actualizada exitosamente', 'success');
                    setTimeout(() => location.reload(), 2000);
                } else {
                    mostrarAlerta(data.message, 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                mostrarAlerta('Error al actualizar la configuración', 'error');
            }
            
            mostrarCargando(false);
        });

        // Agregar cuenta bancaria
        document.getElementById('cuentaForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = {
                banco: document.getElementById('banco').value.trim(),
                numero_cuenta: document.getElementById('numeroCuenta').value.trim(),
                titular: document.getElementById('titular').value.trim(),
                tipo_cuenta: document.getElementById('tipoCuenta').value
            };
            
            if (!formData.banco || !formData.numero_cuenta || !formData.titular) {
                mostrarAlerta('Todos los campos son requeridos', 'error');
                return;
            }
            
            mostrarCargando(true);
            
            try {
                const response = await fetch('api/admin-cuentas.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(formData)
                });
                
                const data = await response.json();
                
                if (data.success) {
                    mostrarAlerta('Cuenta agregada exitosamente', 'success');
                    document.getElementById('cuentaForm').reset();
                    cargarCuentas();
                } else {
                    mostrarAlerta(data.message, 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                mostrarAlerta('Error al agregar la cuenta', 'error');
            }
            
            mostrarCargando(false);
        });

        // Cargar cuentas bancarias
        async function cargarCuentas() {
            try {
                const response = await fetch('api/admin-cuentas.php?action=list');
                const data = await response.json();
                
                if (data.success) {
                    renderizarCuentas(data.data);
                }
            } catch (error) {
                console.error('Error al cargar cuentas:', error);
            }
        }

        // Renderizar cuentas
        function renderizarCuentas(cuentas) {
            const container = document.getElementById('cuentasList');
            
            if (cuentas.length === 0) {
                container.innerHTML = '<p style="text-align: center; color: var(--text-light); padding: 2rem;">No hay cuentas registradas</p>';
                return;
            }
            
            let html = '<table class="table"><thead><tr><th>Banco</th><th>Cuenta</th><th>Titular</th><th>Tipo</th><th>Estado</th><th>Acciones</th></tr></thead><tbody>';
            
            cuentas.forEach(cuenta => {
                html += `
                    <tr>
                        <td>${cuenta.banco}</td>
                        <td>${cuenta.numero_cuenta}</td>
                        <td>${cuenta.titular}</td>
                        <td>${cuenta.tipo_cuenta}</td>
                        <td>
                            <span class="badge ${cuenta.activa ? 'badge-success' : 'badge-danger'}">
                                ${cuenta.activa ? 'Activa' : 'Inactiva'}
                            </span>
                        </td>
                        <td>
                            <button class="btn btn-sm ${cuenta.activa ? 'btn-danger' : 'btn-primary'}" 
                                    onclick="toggleCuenta(${cuenta.id}, ${cuenta.activa})">
                                <i class="fas fa-${cuenta.activa ? 'ban' : 'check'}"></i>
                                ${cuenta.activa ? 'Desactivar' : 'Activar'}
                            </button>
                        </td>
                    </tr>
                `;
            });
            
            html += '</tbody></table>';
            container.innerHTML = html;
        }

        // Toggle estado de cuenta
        async function toggleCuenta(id, estadoActual) {
            mostrarCargando(true);
            
            try {
                const response = await fetch('api/admin-cuentas.php', {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        id: id,
                        activa: !estadoActual
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    mostrarAlerta('Estado de cuenta actualizado', 'success');
                    cargarCuentas();
                } else {
                    mostrarAlerta(data.message, 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                mostrarAlerta('Error al actualizar el estado de la cuenta', 'error');
            }
            
            mostrarCargando(false);
        }

        // Cargar pedidos recientes
        async function cargarPedidos() {
            try {
                const response = await fetch('api/admin-pedidos.php?action=recent&limit=10');
                const data = await response.json();
                
                if (data.success) {
                    renderizarPedidos(data.data);
                }
            } catch (error) {
                console.error('Error al cargar pedidos:', error);
            }
        }

        // Renderizar pedidos
        function renderizarPedidos(pedidos) {
            const tbody = document.querySelector('#pedidosTable tbody');
            
            if (pedidos.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" style="text-align: center; color: var(--text-light); padding: 2rem;">No hay pedidos registrados</td></tr>';
                return;
            }
            
            let html = '';
            pedidos.forEach(pedido => {
                const fecha = new Date(pedido.created_at).toLocaleDateString();
                const estadoBadge = getEstadoBadge(pedido.estado);
                
                html += `
                    <tr>
                        <td>#${pedido.id}</td>
                        <td>${pedido.nombre}</td>
                        <td>${pedido.whatsapp}</td>
                        <td>${pedido.total_boletos}</td>
                        <td>${pedido.monto_total}</td>
                        <td>${estadoBadge}</td>
                        <td>${fecha}</td>
                        <td>
                            <button class="btn btn-sm btn-primary" onclick="verDetallesPedido(${pedido.id})">
                                <i class="fas fa-eye"></i> Ver
                            </button>
                            ${pedido.estado === 'pendiente' ? `
                                <button class="btn btn-sm btn-success" onclick="confirmarPago(${pedido.id})">
                                    <i class="fas fa-check"></i> Pagado
                                </button>
                            ` : ''}
                        </td>
                    </tr>
                `;
            });
            
            tbody.innerHTML = html;
        }

        // Obtener badge de estado
        function getEstadoBadge(estado) {
            switch (estado) {
                case 'pagado':
                    return '<span class="badge badge-success">Pagado</span>';
                case 'pendiente':
                    return '<span class="badge badge-warning">Pendiente</span>';
                case 'cancelado':
                    return '<span class="badge badge-danger">Cancelado</span>';
                default:
                    return '<span class="badge badge-danger">Desconocido</span>';
            }
        }

        // Ver detalles de pedido
        function verDetallesPedido(pedidoId) {
            // Implementar modal o redirección para ver detalles
            alert(`Ver detalles del pedido #${pedidoId}`);
        }

        // Confirmar pago
        async function confirmarPago(pedidoId) {
            if (!confirm('¿Estás seguro de confirmar el pago de este pedido?')) {
                return;
            }
            
            mostrarCargando(true);
            
            try {
                const response = await fetch('api/admin-pedidos.php', {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        id: pedidoId,
                        accion: 'confirmar_pago'
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    mostrarAlerta('Pago confirmado exitosamente', 'success');
                    cargarPedidos();
                    // Recargar estadísticas
                    setTimeout(() => location.reload(), 2000);
                } else {
                    mostrarAlerta(data.message, 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                mostrarAlerta('Error al confirmar el pago', 'error');
            }
            
            mostrarCargando(false);
        }

        // Funciones de utilidad
        function mostrarCargando(mostrar) {
            document.getElementById('loading').style.display = mostrar ? 'block' : 'none';
        }

        function mostrarAlerta(mensaje, tipo) {
            const alerta = document.createElement('div');
            alerta.className = `alert alert-${tipo === 'success' ? 'success' : 'error'}`;
            alerta.innerHTML = `
                <i class="fas fa-${tipo === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
                ${mensaje}
            `;
            
            const container = document.querySelector('.container');
            container.insertBefore(alerta, container.firstChild);
            
            setTimeout(() => {
                if (alerta.parentNode) {
                    alerta.parentNode.removeChild(alerta);
                }
            }, 5000);
            
            alerta.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    </script>
</body>
</html>